% h1 example: IBM's common cryptographic architecture (IBM4758 standard) for hardware security modules (HSW).
%   Copyright (C) 2008 Jean Goubault-Larrecq and LSV, CNRS UMR 8643 & ENS Cachan.
%
%   This file is part of h1.
%
%   h1 is free software; you can redistribute it and/or modify
%   it under the terms of the GNU General Public License as published by
%   the Free Software Foundation; either version 2, or (at your option)
%   any later version.
%
%   h1 is distributed in the hope that it will be useful,
%   but WITHOUT ANY WARRANTY; without even the implied warranty of
%   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%   GNU General Public License for more details.
%
%   You should have received a copy of the GNU General Public License
%   along with h1; see the file COPYING.  If not, write to
%   the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

% To encrypt data
% Host -> HSM : {K}_{km + 'data'}, MSG
% HSM -> Host : {MSG}_K
%     where km is master key (should be secret, really long-term secret)
%           K is stored secret key
%           data is a constant (bit mask)
% Here are the possible bit masks:
%	data, export, import, pin, partial
% There is also a key km (master key).

% We encode all messages as msg(M,D,E,I,P,Km,Kp) where M is actual message,
% and D,E,I,P are booleans representing each possible bit mask,
% and Km and Kp are booleans representing whether km or kp is in the message.
% Formally, msg(M,D,E,I,P,Km,Kp) represents M [xor data if D=true] [xor export if E=true] etc.
% If M is not present, we let M=zero.

% First, intruders may change bit masks at will, but does not know km.
cnf(bool_true,axiom,
	(bool(true))).
cnf(bool_false,axiom,
	(bool(false))).
cnf(twiddle_bits,axiom,
	(~bool(D') | ~bool(E') | ~bool(I') | ~bool(P') | ~bool(Kp')
        | ~knows(msg(M,D',E',I',P',Km,Kp'))
	| knows(msg(M,D,E,I,P,Km,Kp)))).

% Then, intruder knows zero:
cnf(knows_zero,axiom,
	(knows(msg(zero,false,false,false,false,false,false)))).

% Standard Dolev-Yao intruder rules, with the msg(...) trick around
% each subterm (except nil and cons, which are only used to build
% intermediate lists).
cnf(intruder_knows_nil,axiom,
  (knows(nil))).
cnf(intruder_can_take_first_components,axiom,
  (~knows(cons(M1,M2)) | knows(M1))).
cnf(intruder_can_take_second_components,axiom,
  (~knows(cons(M1,M2)) | knows(M2))).
cnf(intruder_can_build_pairs,axiom,
  (~knows(M2) | ~knows(M1) | knows(cons(M1,M2)))).
cnf(intruder_can_encrypt,axiom,
  (~knows(K) | ~knows(M)
  | knows(msg(crypt(M,K),false,false,false,false,false,false)))).
% There are only symmetric keys here.
cnf(intruder_can_decrypt,axiom,
  (~knows(K) | ~knows(msg(crypt(M,K),false,false,false,false,false,false))
  | knows(M))).

cnf(host_agent,axiom,
    (agent(host))).
cnf(intruder_agent,axiom,
    (agent(i))).

% The intruder can also do exclusive ors.  We only need to consider to consider
% the M,Km, and Kp parts of msg:
cnf(xor1_00,axiom,
	(~bool(X) | xor1(X,X,false))).
cnf(xor1_01,axiom,
	(xor1(false,true,true))).
cnf(xor1_10,axiom,
	(xor1(true,false,true))).
cnf(xor2_00,axiom,
	(~xor1(B1,C1,X1)
	| ~bool(X)
	| xor2(X,B1,X,C1,false,X1))).
cnf(xor2_01,axiom,
	(~xor1(B1,C1,X1)
	| xor2(false,B1,true,C1,true,X1))).
cnf(xor2_10,axiom,
	(~xor1(B1,C1,X1)
	| xor2(true,B1,false,C1,true,X1))).

cnf(xor3_00,axiom,
	(~xor2(B1,B2,C1,C2,X1,X2)
	| ~bool(X)
	| xor3(X,B1,B2,false,C1,C2,X,X1,X2))).
cnf(xor3_01,axiom,
	(~xor2(B1,B2,C1,C2,X1,X2)
	| xor3(false,B1,B2,true,C1,C2,true,X1,X2))).
cnf(xor3_10,axiom,
	(~xor2(B1,B2,C1,C2,X1,X2)
	| xor3(true,B1,B2,false,C1,C2,true,X1,X2))).

cnf(xor4_00,axiom,
	(~xor3(B1,B2,B3,C1,C2,C3,X1,X2,X3)
	| ~bool(X)
	| xor4(X,B1,B2,B3,false,C1,C2,C3,X,X1,X2,X3))).
cnf(xor4_01,axiom,
	(~xor3(B1,B2,B3,C1,C2,C3,X1,X2,X3)
	| xor4(false,B1,B2,B3,true,C1,C2,C3,true,X1,X2,X3))).
cnf(xor4_10,axiom,
	(~xor3(B1,B2,B3,C1,C2,C3,X1,X2,X3)
	| xor4(true,B1,B2,B3,false,C1,C2,C3,true,X1,X2,X3))).

cnf(xor5_00,axiom,
	(~xor4(B1,B2,B3,B4,C1,C2,C3,C4,X1,X2,X3,X4)
	| ~bool(X)
	| xor5(X,B1,B2,B3,B4,false,C1,C2,C3,C4,X,X1,X2,X3,X4))).
cnf(xor5_01,axiom,
	(~xor4(B1,B2,B3,B4,C1,C2,C3,C4,X1,X2,X3,X4)
	| xor5(false,B1,B2,B3,B4,true,C1,C2,C3,C4,true,X1,X2,X3,X4))).
cnf(xor5_10,axiom,
	(~xor4(B1,B2,B3,B4,C1,C2,C3,C4,X1,X2,X3,X4)
	| xor5(true,B1,B2,B3,B4,false,C1,C2,C3,C4,true,X1,X2,X3,X4))).

cnf(xor6_00,axiom,
	(~xor5(B1,B2,B3,B4,B5,C1,C2,C3,C4,C5,X1,X2,X3,X4,X5)
	| ~bool(X)
	| xor6(X,B1,B2,B3,B4,B5,false,C1,C2,C3,C4,C5,X,X1,X2,X3,X4,X5))).
cnf(xor6_01,axiom,
	(~xor5(B1,B2,B3,B4,B5,C1,C2,C3,C4,C5,X1,X2,X3,X4,X5)
	| xor6(false,B1,B2,B3,B4,B5,true,C1,C2,C3,C4,C5,true,X1,X2,X3,X4,X5))).
cnf(xor6_10,axiom,
	(~xor5(B1,B2,B3,B4,B5,C1,C2,C3,C4,C5,X1,X2,X3,X4,X5)
	| xor6(true,B1,B2,B3,B4,B5,false,C1,C2,C3,C4,C5,true,X1,X2,X3,X4,X5))).

cnf(xor_00,axiom,
	(~xor6(B1,B2,B3,B4,B5,B6,C1,C2,C3,C4,C5,C6,X1,X2,X3,X4,X5,X6)
	| xor(M,B1,B2,B3,B4,B5,B6,M,C1,C2,C3,C4,C5,C6,zero,X1,X2,X3,X4,X5,X6))).
cnf(xor_01,axiom,
	(~xor6(B1,B2,B3,B4,B5,B6,C1,C2,C3,C4,C5,C6,X1,X2,X3,X4,X5,X6)
	| xor(zero,B1,B2,B3,B4,B5,B6,M,C1,C2,C3,C4,C5,C6,M,X1,X2,X3,X4,X5,X6))).
cnf(xor_10,axiom,
	(~xor6(B1,B2,B3,B4,B5,B6,C1,C2,C3,C4,C5,C6,X1,X2,X3,X4,X5,X6)
	| xor(M,B1,B2,B3,B4,B5,B6,zero,C1,C2,C3,C4,C5,C6,M,X1,X2,X3,X4,X5,X6))).

cnf(knows_xor,axiom,
    (~knows(msg(M,D,E,I,P,Km,Kp))
    | ~knows(msg(M',D',E',I',P',Km',Kp'))
    | ~xor(M,D,E,I,P,Km,Kp,M',D',E',I',P',Km',Kp',M'',D'',E'',I'',P'',Km'',Kp'')
    | knows(msg(M'',D'',E'',I'',P'',Km'',Kp'')))).


% To encrypt data
% Host -> HSM : {K}_{km + 'data'}, MSG
% HSM -> Host : {MSG}_K
% We ignore all parity checks.
cnf(encrypt_data_command,axiom,
    (~knows(crypt(K,
                  msg(zero,true,false,false,false,true,false)), % km+'data'
    | ~knows(Msg)
    | knows(crypt(Msg,K))).

% To export a key:
% Host -> HSM : {KEK}_{km + 'exp'}, 'data', {d1}_{km + 'data'}
% HSM -> Host : {d1}_{KEK + data}
% where KEK is the key exporting key
%  The type 'data' is not checked.
cnf(export_key_command,axiom,
	(~knows(msg(crypt(msg(KEK_M,KEK_D,KEK_E,KEK_I,KEK_P,KEK_Km,KEK_Kp),
	                  msg(zero,false,true,false,false,true,false)),
                    false,false,false,false,false,false))
        | ~knows(M,D,E,I,P,Km,Kp) % should be 'data', but can be anything
        | ~knows(msg(crypt(D1,
	                   msg(zero,true,false,false,false,false,true,false)),
                     false,false,false,false,false,false))
        | xor(KEK_M,KEK_D,KEK_E,KEK_I,KEK_P,KEK_Km,KEK_Kp,
              M,D,E,I,P,Km,Kp,
              M',D',E',I',P',Km',Kp')
        | knows(msg(crypt(D1,
                          msg(M',D',E',I',P',Km',Kp')),
                    false,false,false,false,false,false)))).

%!!! unfinished.

cnf(intruder_knows_all_agents,axiom,
  (~agent(X) | knows(X))).
% We assume the intruder knows at least one data key {c}_{km + 'data'}:
cnf(intruder_knows_a_data_key,axiom,
  (knows(msg(crypt(msg(c,false,false,false,false,false,false),
                   msg(zero,true,false,false,false,true,false)),
             false,false,false,false,false,false)))).
